{******************************************************************************
 ******************************************************************************
 *
 *
 *  EZ-KIT Lite resident monitor
 *
 *  This file is part of the ADSP 2181 EZ-KIT Lite resident monitor.
 *
 *  The monitor is loaded from the on-board EPROM through the BDMA
 *  interface during power up.
 *
 *  The complete source codes include the following files:
 *
 *  INTVEC  .DSP : interrupt vector table
 *  UART    .DSP : UART simulation routine
 *  MAIN    .DSP : main module
 *  SELFTEST.DSP : power up self test routine
 *  UTILITY .DSP : send a line of ASCII text through serial line
 *  COMMAND .DSP : command loop and command interpreter
 *  CODEC   .DSP : 1847 codec initialization and interrupt handling
 *
 *  TONE    .DAT : 32 samples of one complete cycle sine wave
 *  CONSTANT.K   : constants
 *
 *
 ******************************************************************************
 ******************************************************************************}










.module/ram     SELFTEST;
.include    <constant.k>;






.entry selftest;




.var/ram selfTstRst;             { self test result. }

{   The following variables represent user memories. }
.var/rom/pm         pm_user [0x37d0];
.var/rom/dm         dm_user [0x3e00];




.global             selfTstRst;


{******************************************************************************
 *
 *  Performs the following tests:
 *      1.  reset register states test
 *      2.  DSP register test
 *      3.  do loop test
 *      4.  on-chip program and data memory test
 *      5.  1847 codec test
 *
 *
 *  Result bit codes :
 *
 *  0x0001 = reset memory mapped register state test failed.
 *  0x0002 = DSP registers test failed.
 *  0x0004 = do loop test failed.
 *  0x0008 = program and data memory test failed
 *  0x0010 = codec initialization failed
 *
 *
 *  REGISTER USAGE SUMMARY:
 *
 *  input  :
 *  modyfy :
 *  output :
 *  destroy:
 *  keep   :
 *  memory :
 *  calls  :
 *
 ******************************************************************************}

selftest:


        ar = 0;
        dm (selfTstRst) = ar;


{=== reset state test ======================================================}
{   checking register contents agains known reset state value               }
        ax0 = 0x0000;   ay0 = astat;
        none = ax0 - ay0;
        if ne jump test1bad;

        ax0 = 0x0054;   ay0 = sstat;
        none = ax0 - ay0;
        if ne jump test1bad;

        ax0 = 0x0000;   ay0 = dm (System_Control_Reg);
        none = ax0 - ay0;
        if ne jump test1bad;

        ax0 = 0x7fff;   ay0 = dm (DM_Wait_Reg);
        none = ax0 - ay0;
        if ne jump test1bad;

        ay0 = dm (SPORT0_Control_Reg);
        none = pass  ay0;
        if ne jump test1bad;

        ax0 = 0xf003;   ay0 = dm (SPORT0_Autobuf);
        ar  = ax0 and ay0;
        none = pass ar;
        if ne jump test1bad;

        ax0 = 0x7fff;   ay0 = dm (SPORT1_Control_Reg);
        ar  = ax0 and ay0;
        none = pass ar;
        if ne jump test1bad;

        ax0 = 0x7f00;   ay0 = dm (PFTYPE);
        none = ax0 - ay0;
        if ne jump test1bad;

        {   by viture of these codes being run, BDMA registers must be ok }

        jump test1ok;
test1bad:
        ax0 = 0x0001;
        ay0 = dm (selfTstRst);
        ar = ax0 or ay0;
        dm (selfTstRst) = ar;
test1ok:





{=== DSP registers test ====================================================}
{   reading and writing DSP registers                                       }
        ax0 = 0xffff;
        call copyReg;
        ay0 = si;
        none = ax0 - ay0;
        if ne jump test2bad;
        ax0 = 0x3fff;
        ay0 = l7;
        none = ax0 - ay0;
        if ne jump test2bad;
        ax0 = 0xff;
        ay0 = px;
        none = ax0 - ay0;
        if ne jump test2bad;
        ax0 = 0xffff;
        ay0 = sb;
        none = ax0 - ay0;
        if ne jump test2bad;


        ax0 = 0x5555;
        call copyReg;
        ay0 = si;
        none = ax0 - ay0;
        if ne jump test2bad;
        ax0 = 0x1555;
        ay0 = l7;
        none = ax0 - ay0;
        if ne jump test2bad;
        ax0 = 0x55;
        ay0 = px;
        none = ax0 - ay0;
        if ne jump test2bad;
        ax0 = 0xfff5;
        ay0 = sb;
        none = ax0 - ay0;
        if ne jump test2bad;


        ax0 = 0xaaaa;
        call copyReg;
        ay0 = si;
        none = ax0 - ay0;
        if ne jump test2bad;
        ax0 = 0x2aaa;
        ay0 = l7;
        none = ax0 - ay0;
        if ne jump test2bad;
        ax0 = 0xaa;
        ay0 = px;
        none = ax0 - ay0;
        if ne jump test2bad;
        ax0 = 0x0a;
        ay0 = sb;
        none = ax0 - ay0;
        if ne jump test2bad;


        ax0 = 0x0000;
        call copyReg;
        ay0 = si;
        none = pass ay0;
        if ne jump test2bad;
        ay0 = l7;
        none = pass ay0;
        if ne jump test2bad;
        ay0 = px;
        none = pass ay0;
        if ne jump test2bad;
        ay0 = sb;
        none = pass ay0;
        if ne jump test2bad;


{-- test pcovlay }
{   better not swap our world in memory away.
    DO NOT CHANGE PCOVLAY AS IT SWAP OUT MONITOR CODES.
        ar = 0;
        pmovlay = ar;
        ay0 = pmovlay;
        none = ar - ay0;
        if ne jump test2bad;

        ar = 1;
        pmovlay = ar;
        ay0 = pmovlay;
        none = ar - ay0;
        if ne jump test2bad;

        ar = 2;
        pmovlay = ar;
        ay0 = pmovlay;
        none = ar - ay0;
        if ne jump test2bad;
}

{-- test dcovlay }
        ar = 0;
        dmovlay = ar;
        ay0 = dmovlay;
        none = ar - ay0;
        if ne jump test2bad;

        ar = 1;
        dmovlay = ar;
        ay0 = dmovlay;
        none = ar - ay0;
        if ne jump test2bad;

        ar = 2;
        dmovlay = ar;
        ay0 = dmovlay;
        none = ar - ay0;
        if ne jump test2bad;


        jump test2ok;
test2bad:
        ax0 = 0x0002;
        ay0 = dm (selfTstRst);
        ar = ax0 or ay0;
        dm (selfTstRst) = ar;
test2ok:
        ar = 0;
        pmovlay = ar;
        dmovlay = ar;


{=== Do loop control test ==================================================}
{   Do four level deep do loop and check number of executions               }
        ar = 0;
        cntr = 15;
        do lb1 until ce;
            cntr = 15;
            do lb2 until ce;
                cntr = 15;
                do lb3 until ce;
                    cntr = 15;
                    do lb4 until ce;
                        ar = ar + 1;
lb4:                nop;
lb3:            nop;
lb2:        nop;
lb1:    nop;
        ay0 = 50625;
        none = ar - ay0;
        if ne jump test3bad;

        jump test3ok;
test3bad:
        ax0 = 0x0004;
        ay0 = dm (selfTstRst);
        ar = ax0 or ay0;
        dm (selfTstRst) = ar;
test3ok:



{=== program and data memory test ==========================================}
{   write known patterns to unused program and data program and verify      }
{   array pm_user and dm_user occupy unused memories                        }
{   pm }
        i4 = ^pm_user;
        m4 = 1;
        cntr = %pm_user;
        do lb10 until ce;
        ar = i4;
lb10:   pm (i4, m4) = ar;

        i4 = ^pm_user;
        cntr = %pm_user;
        do lb11 until ce;
        ar = i4;
        ay0 = pm (i4, m4);
        none = ar - ay0;
        if eq jump lb11;
lb10a:
      { pop cntr; }         { no need to pop counter because it's empty. }
        pop pc;                     { popping relavent stacks since we }
        pop loop;                   { are jumping out of do loop. }
        jump test4bad;
lb11:   nop;


{   dm }
        i4 = ^dm_user;
        cntr = %dm_user;
        do lb12 until ce;
        ar = i4;
lb12:   dm (i4, m4) = ar;

        i4 = ^dm_user;
        cntr = %dm_user;
        do lb13 until ce;
        ar = i4;
        ay0 = dm (i4, m4);
        none = ar - ay0;
        if ne jump lb10a;           { jump to pop stacks }
lb13:   nop;

        jump test4ok;

test4bad:
        ax0 = 0x0008;
        ay0 = dm (selfTstRst);
        ar = ax0 or ay0;
        dm (selfTstRst) = ar;
test4ok:



{=== 1847 test =============================================================}
{   initialize 1847 codec and verify all codec register values              }
        ena ints;

        call codecInit;
        cntr = 400;
        do lb31 until ce;
        idle;
lb31:   nop;


        ax0 = 0x2000; dm (tx_buf) = ax0; idle;
        ax0 = 0x2100; dm (tx_buf) = ax0; idle;
        ax0 = 0x2200; dm (tx_buf) = ax0; idle; ax0 = dm (rx_buf);{dm (0x2000) = ax0;}ay0 = 0x2002; none = ax0 - ay0; if ne jump test5bad;
        ax0 = 0x2300; dm (tx_buf) = ax0; idle; ax0 = dm (rx_buf);{dm (0x2001) = ax0;}ay0 = 0x2102; none = ax0 - ay0; if ne jump test5bad;
        ax0 = 0x2400; dm (tx_buf) = ax0; idle; ax0 = dm (rx_buf);{dm (0x2002) = ax0;}ay0 = 0x2288; none = ax0 - ay0; if ne jump test5bad;
        ax0 = 0x2500; dm (tx_buf) = ax0; idle; ax0 = dm (rx_buf);{dm (0x2003) = ax0;}ay0 = 0x2388; none = ax0 - ay0; if ne jump test5bad;
        ax0 = 0x2600; dm (tx_buf) = ax0; idle; ax0 = dm (rx_buf);{dm (0x2004) = ax0;}ay0 = 0x2488; none = ax0 - ay0; if ne jump test5bad;
        ax0 = 0x2700; dm (tx_buf) = ax0; idle; ax0 = dm (rx_buf);{dm (0x2005) = ax0;}ay0 = 0x2588; none = ax0 - ay0; if ne jump test5bad;
        ax0 = 0x2800; dm (tx_buf) = ax0; idle; ax0 = dm (rx_buf);{dm (0x2006) = ax0;}ay0 = 0x2600; none = ax0 - ay0; if ne jump test5bad;
        ax0 = 0x2900; dm (tx_buf) = ax0; idle; ax0 = dm (rx_buf);{dm (0x2007) = ax0;}ay0 = 0x2700; none = ax0 - ay0; if ne jump test5bad;
        ax0 = 0x2a00; dm (tx_buf) = ax0; idle; ax0 = dm (rx_buf);{dm (0x2008) = ax0;}ay0 = 0x285b; none = ax0 - ay0; if ne jump test5bad;
        ax0 = 0x2c00; dm (tx_buf) = ax0; idle; ax0 = dm (rx_buf);{dm (0x2009) = ax0;}ay0 = 0x2909; none = ax0 - ay0; if ne jump test5bad;
        ax0 = 0x2d00; dm (tx_buf) = ax0; idle; ax0 = dm (rx_buf);{dm (0x200a) = ax0;}ay0 = 0x2a00; none = ax0 - ay0; if ne jump test5bad;
                                         idle; ax0 = dm (rx_buf);{dm (0x200b) = ax0;}ay0 = 0x2c40; none = ax0 - ay0; if ne jump test5bad;
                                         idle; ax0 = dm (rx_buf);{dm (0x200c) = ax0;}ay0 = 0x2d00; none = ax0 - ay0; if ne jump test5bad;

        jump test5ok;


test5bad:
        ax0 = 0x0010;
        ay0 = dm (selfTstRst);
        ar = ax0 or ay0;
        dm (selfTstRst) = ar;
test5ok:

        ax0 = 0xaf00;
        dm (tx_buf + 0) = ax0;


        { turn off transmit interrupt }
        ax0 = imask;
        ay0 = b#1110111111;
          {     |||||||||+ | timer
                ||||||||+- | SPORT1 rec or IRQ0
                |||||||+-- | SPORT1 trx or IRQ1
                ||||||+--- | BDMA
                |||||+---- | IRQE
                ||||+----- | SPORT0 rec
                |||+------ | SPORT0 trx
                ||+------- | IRQL0
                |+-------- | IRQL1
                +--------- | IRQ2
          }
        ar = ax0 and ay0;
        imask = ar;


        ay0 = dm (selfTstRst);


        rts;



{******************************************************************************
 *  This sub-routine is used by the DSP register test.  AX0 is set to the
 *  desireable pattern and passed through all registers to be tested.
 *  The last register in the chain is si (16 bits,) l7 (14 bits,) px (8
 *  bits,) and sb (5 bits).
 *
 *
 *  16 bits:
 *  ax0, ax1, ay0, ay1, af, ar
 *  mx0, mx1, my0, my1, mr0, mr1, mf
 *  si, sr0, sr1
 *
 *  14 bits;
 *  i0-7, m0-7, l0-7
 *
 *  8 bits:
 *  mr2
 *  se
 *  px
 *
 *  5 bits:
 *  sb
 *
 *  2 bits:
 *  pmovlay, dmovlay
 *
 ******************************************************************************}

copyReg:

        ax1 = ax0;
        ar  = ax1;
        mr0 = ar;
        mr1 = mr0;
        sr0 = mr1;
        sr1 = sr0;
        ay0 = sr1;
        ay1 = ay0;
        af  = pass ay1;
        ar = pass af;
        mx0 = ar;
        mx1 = mx0;
        my0 = mx1;
        my1 = my0;
        si  = my1;

        i0 = si;
        i1 = i0;
        i2 = i1;
        i3 = i2;
        i4 = i3;
        i5 = i4;
        i6 = i5;
        i7 = i6;
        m0 = i7;
        m1 = m0;
        m2 = m1;
        m3 = m2;
        m4 = m3;
        m5 = m4;
        m6 = m5;
        m7 = m6;
        l0 = m7;
        l1 = l0;
        l2 = l1;
        l3 = l2;
        l4 = l3;
        l5 = l4;
        l6 = l5;
        l7 = l6;

        mr2 = l7;
        se = mr2;
        px = se;

        sb = px;

        rts;


.endmod;
